//==============================================================================
// Project:   Wii C++ Template
//==============================================================================

/**
 * @mainpage Wii C++ Console/Skins Library
 * These libraries provide skinned console applications.
 */
//------------------------------------------------------------------------------
// Headers
//------------------------------------------------------------------------------
#include <stdlib.h>
#include <stdio.h>
#include <gccore.h>
#include <wiiuse/wpad.h>

#include "console/console.h"
#include "io/FileIO.h"

#include "skins/SkinController.h"


#define SYS_RETURNTOHBMENU   7

//------------------------------------------------------------------------------
// Externals
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Globals
//------------------------------------------------------------------------------
static GXRModeObj *rmode = NULL;
u8 HWButton = 0;
static void *xfb = NULL;

/**
 * Callback for the reset button on the Wii.
 */
void WiiResetPressed()
{
	HWButton = SYS_RETURNTOMENU;
}

/**
 * Callback for the power button on the Wii.
 */
void WiiPowerPressed()
{
	HWButton = SYS_POWEROFF_STANDBY;
}

/**
 * Callback for the power button on the Wiimote.
 * @param[in] chan The Wiimote that pressed the button
 */
void WiimotePowerPressed(s32 chan)
{
	HWButton = SYS_POWEROFF_STANDBY;
	//SYS_POWEROFF
}

//_______________________________________________________________________________
/**
 * Entry point.
 * @param[in] argc The number of arguments invoked with the program
 * @param[in] argv The array containing the arguments
 * @return 0 on clean exit, an error code otherwise
 */
int main(int argc, char **argv)
{
		// Initialise the video system
	VIDEO_Init();
	
	// This function initialises the attached controllers
	WPAD_Init();
	
	// Obtain the preferred video mode from the system
	// This will correspond to the settings in the Wii menu
	rmode = VIDEO_GetPreferredMode(NULL);

	// Allocate memory for the display in the uncached region
	xfb = MEM_K0_TO_K1(SYS_AllocateFramebuffer(rmode));
	
	// Initialise the console, required for printf
	console_init(xfb,20,20,rmode->fbWidth,rmode->xfbHeight,rmode->fbWidth*VI_DISPLAY_PIX_SZ);
	
	// Set up the video registers with the chosen mode
	VIDEO_Configure(rmode);
	
	// Tell the video hardware where our display memory is
	VIDEO_SetNextFramebuffer(xfb);
	
	// Make the display visible
	VIDEO_SetBlack(FALSE);

	// Flush the video register changes to the hardware
	VIDEO_Flush();

	// Wait for Video setup to complete
	VIDEO_WaitVSync();
	if(rmode->viTVMode&VI_NON_INTERLACE) VIDEO_WaitVSync();


	
	//Initialise the fat subsystem
	FileIO io;
	
	//io.initialiseFat();
	
	//Now an instance of the skins class passing the app path so
	//it knows where to locate the skin file.  Can construct without this and
	//it'll use relative addressing, which works fine from the HBC but not so well
	//when loaded from a channel or other loader such as LoadMii.
	SkinController skins;
	
	
	//Get an instance of the console class
	Console myConsole;
	myConsole.registerSkin(skins.getSkin(0));
	
	sleep(10);
	myConsole.clearConsole();
	
	int startY=5;
	int onSkin = 0;
	bool skinChange = true;
	
	
	
	while(1) 
	{

		//If the skin has changed (or first time in) then redraw the windows and text!
		if(skinChange==true)
		{
			myConsole.registerSkin(skins.getSkin(onSkin));
			//Clear the console screen - could also use clear area to clear specific areas
			myConsole.clearConsole();
			
			myConsole.drawString(02,startY,skins.getSkin(onSkin).getSkinConsoleTextColor(),skins.getSkin(onSkin).getSkinConsoleBgColor(),"Showing Skin:");
			myConsole.drawString(18,startY,skins.getSkin(onSkin).getSkinConsoleTextColor(),skins.getSkin(onSkin).getSkinConsoleBgColor(),skins.getSkin(onSkin).getSkinName());
			
			
			//Draw Primary Window and some text
			myConsole.drawPrimaryWindow(5,startY+3,6,70,"Test Primary Window");
			myConsole.drawStringPrimary(7,startY+5,"Push Left to move to next skin");
			myConsole.drawStringPrimaryAlternate(7,startY+7,"Push Right to return to previous skin");
	
			//Draw Secondary Window and some text
			myConsole.drawSecondaryWindow(5,startY+13,6,70,"Test Secondary Window");
			myConsole.drawStringSecondary(7,startY+15,"Press HOME to exit");
			myConsole.drawStringSecondaryAlternate(7,startY+17,"Press HOME to exit");			
			
			skinChange = false;
		}
		// Call WPAD_ScanPads each loop, this reads the latest controller states
		WPAD_ScanPads();

	
		// WPAD_ButtonsDown tells us which buttons were pressed in this loop
		// this is a "one shot" state which will not fire again until the button has been released
		u32 pressed = WPAD_ButtonsDown(0);
		u16 gcPressed = PAD_ButtonsDown(0);

		// We return to the launcher application via exit
		if ( pressed & WPAD_BUTTON_HOME ) exit(0);
		
		// If push right on Wii/GC Pad then change skin!
		if ( (pressed & WPAD_BUTTON_RIGHT) || (PAD_StickX(0) > 18) || (gcPressed & PAD_BUTTON_RIGHT) || (gcPressed & PAD_TRIGGER_R))
		{
			onSkin++;
			if(onSkin >= skins.getNumberSkins())onSkin=0;
			skinChange = true;
		}
		
		// If push left on Wii/GC Pad then change skin!
		if ( (pressed & WPAD_BUTTON_LEFT) || (PAD_StickX(0) < -18) || (gcPressed & PAD_BUTTON_LEFT) || (gcPressed & PAD_TRIGGER_L))
		{
			onSkin--;
			if(onSkin <0) onSkin = (skins.getNumberSkins()-1);
			skinChange = true;
		}

		// Wait for the next frame
		VIDEO_WaitVSync();
	}
	return 0;
}
