/**
 *
 * Skinned Console Applications C++/Wii Library
 * (C)2009 http://www.pembo.co.uk
 *
 **/
 
//------------------------------------------------------------------------------
// Headers
//------------------------------------------------------------------------------
# include "console.h"


//------------------------------------------------------------------------------
// Externals
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Globals
//------------------------------------------------------------------------------



//_______________________________________________________________________________
/**
 * Constructor
 */
 Console::Console()
 {
	skinRegistered = false;
 }
 
//_______________________________________________________________________________
/**
 * destructor
 */
 Console::~Console()
 {
	skinRegistered = false;
 }
 
//_______________________________________________________________________________
/**
 * Gets the actual console color code from the MACRO
 * @param[in] color The number of the colour to translate.
 *
 */
int Console::translateColor(int color)
{
	if(color>=10) color = color -10;
	return color;
}

//_______________________________________________________________________________
/**
 * Returns TRUE of FALSE if the color is bright version or not
 * @param[in] color The number of the colour to test for brightness
 * @return isBrightColor Whether the colour is  bright version or not
 *
 */
bool Console::isBrightColor(int color)
{
	if(color>=10)
	{
		return true;
	}
	else
	{
		return false;
	}
}


//_______________________________________________________________________________
/**
 * Clears the console setting the background with the specified colour
 * @param[in] color to clear console with
 */
void Console::clearConsole(int bgcolor)
{
	int setColor = translateColor(bgcolor);
	
	if(isBrightColor(bgcolor)==true)
	{
		printf("\x1b[4%i;1m",setColor);
	}
	else
	{
		printf("\x1b[4%im",setColor);
	}
	printf("\x1b[2J");
}

//_______________________________________________________________________________
/**
 * Clears a specific area of the screen with the provided colour
 * used (to prevent tearing) when constantly updating a screen
 * @param[in] startX Starting X (cols) position of area to clear
 * @param[in] startY Starting Y (rows) position of area to clear
 * @param[in] rows The number of rows to clear (Y)
 * @param[in] cols The number of columns to clear (X)
 * @param[in] bgcolor The colour to clear the area with
 */
void Console::clearArea(int startX,int startY, int rows, int cols, int bgcolor)
{

	int i,j;
	for(i=0;i<cols;i++)
	{
	  for(j=1;j<rows;j++)
	  {
		drawString(startX+i,startY+j,bgcolor,bgcolor," ");
	  }
	}
	
}

//_______________________________________________________________________________
/**
 * prints a backspace character
 */
void Console::backspace()
{
	printf("\b");
}

//_______________________________________________________________________________
/**
 * Moves the cursor to the provided row / column on the screen
 * @param[in] row The row (Y) position to move the cursor to.
 * @param[in] column The column (X) position to move the cursor to.
 */
void Console::moveCursor (int row, int column)
{
	printf("\x1b[%i;%iH",row,column);
}

//_______________________________________________________________________________
/**
 * Sets the Colour of the console text
 * @param[in] fgcolor The colour to set the foreground (text) to.
 * @param[in] bgcolor The colour to set the text background to.
 */
void Console::setColor(int fgcolor,int bgcolor)
{
	//Get for brightness
	int fgBright = isBrightColor(fgcolor);
	int bgBright = isBrightColor(bgcolor);
	
	//Translate colours
	int actualFgColor = translateColor(fgcolor);
	int actualBgColor = translateColor(bgcolor);

	//Set colours
	if(bgBright==true)
	{
		printf("\x1b[4%i;1m",actualBgColor);
	}
	else
	{
		printf("\x1b[4%im",actualBgColor);
	}	
	
	if(fgBright==true)
	{
		printf("\x1b[3%i;1m",actualFgColor);
	}
	else
	{
		printf("\x1b[3%im",actualFgColor);
	}
}

//_______________________________________________________________________________
/**
 * Draws a text string at a given position in the provided color and intensity
 * @param[in] x The X position at which to draw the text (cols)
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] color The colour to set the text background to.
 * @param[in] bgcolor The colour to set the text background to.
 * @param[in] msg Message which to draw onto the screen
 */
void Console::drawString (int x, int y, int color, int bgcolor, const char *msg)
{
	moveCursor(y,x);
	setColor(color,bgcolor);
	printf(msg);
	setColor(WHITE,BLACK);
}

//_______________________________________________________________________________
/**
 * Draws a text string with the center of the screen at a given Y position in the 
 * provided color and intensity
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] color The colour to set the text background to.
 * @param[in] bgcolor The colour to set the text background to.
 * @param[in] msg Message which to draw onto the screen
 */
void Console::centerString (int y, int color, int bgcolor, const char *msg)
{

	int stringLen = strlen(msg);
	int centrePos = 40;
	int titleLenDiv2 = stringLen / 2;
	int x = (centrePos - titleLenDiv2);
	drawString(x,y,color,bgcolor,msg);
}


//_______________________________________________________________________________
/**
 * Draws a console based window
 * @param[in] startX starting X (cols) position of the window
 * @param[in] startY starting Y (rows) position of the window
 * @param[in] rows height of the window (rows)
 * @param[in] cols width of the window (cols)
 * @param[in] bgColor Background color of the window
 * @param[in] title_textColor Colour of the title text
 * @param[in] title_textlineColor Colour of the title line
 * @param[in] title_bgColor Background colour of the window top
 * @param[in] shadowColor Shadow Colour
 * @param[in] declColor Colour of the left/right/bottom window decals
 * @param[in] windowTitle Name of the window to display in the title
  */
void Console::drawWindow (int startX,int startY,int rows,int cols, int bgColor,int title_textColor,int title_textlineColor,int title_bgColor, int shadowColor, int declColor, const char* windowTitle)
{
	int i=0;
	int j=0;
	
	for(i=0;i<cols;i++)
	{
		drawString(startX+i,startY,title_bgColor,title_bgColor," ");
	}
	
	int titleLen = strlen(windowTitle);
	int centrePos = cols/2+startX;
	int titleLenDiv2 = titleLen / 2;
	
	int titlePos = (centrePos - titleLenDiv2);
	int titlePosMinusTwo = (titlePos -2);
	
	
	/** draw the lines **/
	i=0;
	for(i=startX;i<titlePosMinusTwo;i++)
	{
		drawString(i,startY,title_textlineColor,title_bgColor,"-");
	}
	drawString(titlePosMinusTwo,startY,title_textlineColor,title_bgColor,"|");

	drawString(titlePos,startY,title_textColor,title_bgColor,windowTitle);

	drawString(titlePos + titleLen + 1,startY,title_textlineColor,title_bgColor,"|");
	i=0;
	for(i=titlePos + titleLen + 2;i<cols+startX;i++)
	{
		drawString(i,startY,title_textlineColor,title_bgColor,"-");
	}
	
	
	for(i=0;i<cols;i++)
	{
	  for(j=1;j<rows;j++)
	  {
		drawString(startX+i,startY+j,bgColor,bgColor," ");
	  }
	}
	
	//bottom line
	for(i=0;i<cols;i++)
	{
		drawString(startX+i,startY+rows-1,declColor,bgColor,"-");
	}
	
	//left & right line
	for(j=1;j<rows-1;j++)
	{
		drawString(startX,startY+j,declColor,bgColor,"|");
		drawString(startX+cols-1,startY+j,declColor,bgColor,"|");
	}
	
	//Corners can be enabled here (with +s) however they don't look so
	//goo so are disabled by default
	//Corners
	//top left
	//drawString(startX,startY+1,title_textlineColor,bgColor,"+");
	//top right
	//drawString(startX+cols-1,startY+1,title_textlineColor,bgColor,"+");
	//bottom left
	//drawString(startX,startY+rows-1,title_textlineColor,bgColor,"+");
	//bottom right
	//drawString(startX+cols-1,startY+rows-1,title_textlineColor,bgColor,"+");
	
	
	//finally the shadow
	for(i=1;i<=rows;i++)
	{
		drawString(startX+cols,startY+i,shadowColor,shadowColor," ");
	}
	
	for(i=1;i<=cols;i++)
	{
		drawString(startX+i,startY+rows,shadowColor,shadowColor," ");
	}
	
	moveCursor(startY + cols + 3,2);
}



//_______________________________________________________________________________
/**
 * Draws a console based window using skin (primary colours)
 * @param[in] startX starting X (cols) position of the window
 * @param[in] startY starting Y (rows) position of the window
 * @param[in] rows height of the window (rows)
 * @param[in] cols width of the window (cols)
 * @param[in] skin skin object used for window drawing
 * @param[in] windowTitle Name of the window to display in the title
  */
void Console::drawPrimaryWindow (int startX,int startY,int rows,int cols, const char* windowTitle)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawWindow(startX,startY,rows,cols,
		registeredSkin.getSkinPrimaryBgColor(),
		registeredSkin.getSkinPrimaryTitleTextColor(),
		registeredSkin.getSkinPrimaryTitleRowColor(),
		registeredSkin.getSkinPrimaryBgColor(),
		registeredSkin.getSkinPrimaryShadow(),
		registeredSkin.getSkinPrimaryDecalsColour(),
		windowTitle);
}

//_______________________________________________________________________________
/**
 * Draws a console based window using skin (secondary colours)
 * @param[in] startX starting X (cols) position of the window
 * @param[in] startY starting Y (rows) position of the window
 * @param[in] rows height of the window (rows)
 * @param[in] cols width of the window (cols)
 * @param[in] skin skin object used for window drawing
 * @param[in] windowTitle Name of the window to display in the title
  */
void Console::drawSecondaryWindow (int startX,int startY,int rows,int cols, const char* windowTitle)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawWindow(startX,startY,rows,cols,
		registeredSkin.getSkinSecondaryBgColor(),
		registeredSkin.getSkinSecondaryTitleTextColor(),
		registeredSkin.getSkinSecondaryTitleRowColor(),
		registeredSkin.getSkinSecondaryBgColor(),
		registeredSkin.getSkinSecondaryShadow(),
		registeredSkin.getSkinSecondaryDecalsColour(),
		windowTitle);
}


//_______________________________________________________________________________
/**
 * Clears the console setting the background with the specified colour
 * @param[in] skin skin to use to determine clearing colour!
 */
void Console::clearConsole()
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	clearConsole(registeredSkin.getSkinConsoleBgColor());
}

//_______________________________________________________________________________
/**
 * Clears a specific area of the screen with the 
 * console colour 
 * from the provided skin.
 * Used (to prevent tearing) when constantly updating a screen
 * @param[in] startX Starting X (cols) position of area to clear
 * @param[in] startY Starting Y (rows) position of area to clear
 * @param[in] rows The number of rows to clear (Y)
 * @param[in] cols The number of columns to clear (X)
 */
void Console::clearConsoleArea(int startX,int startY, int rows, int cols)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	clearArea(startX,startY,rows,cols,registeredSkin.getSkinConsoleBgColor());
}

//_______________________________________________________________________________
/**
 * Clears a specific area of the screen with the 
 * primary window colour 
 * from the provided skin.
 * Used (to prevent tearing) when constantly updating a screen
 * @param[in] startX Starting X (cols) position of area to clear
 * @param[in] startY Starting Y (rows) position of area to clear
 * @param[in] rows The number of rows to clear (Y)
 * @param[in] cols The number of columns to clear (X)
 */
void Console::clearPrimaryArea(int startX,int startY, int rows, int cols)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	clearArea(startX,startY,rows,cols,registeredSkin.getSkinPrimaryBgColor());
}

//_______________________________________________________________________________
/**
 * Clears a specific area of the screen with the 
 * secondary window colour 
 * from the provided skin.
 * Used (to prevent tearing) when constantly updating a screen
 * @param[in] startX Starting X (cols) position of area to clear
 * @param[in] startY Starting Y (rows) position of area to clear
 * @param[in] rows The number of rows to clear (Y)
 * @param[in] cols The number of columns to clear (X)
 */
void Console::clearSecondaryArea(int startX,int startY, int rows, int cols)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	clearArea(startX,startY,rows,cols,registeredSkin.getSkinSecondaryBgColor());
}


//_______________________________________________________________________________
/**
 * Draws a text string at a given position using colours from the provided skin
 * using the main text colour from the primary window
 * @param[in] x The X position at which to draw the text (cols)
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] msg Message which to draw onto the screen
 */
void Console::drawStringPrimary (int x, int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawString(x,y,registeredSkin.getSkinPrimaryTextColor(),registeredSkin.getSkinPrimaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string at a given position using colours from the provided skin
 * using the alternate text colour from the primary window
 * @param[in] x The X position at which to draw the text (cols)
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] msg Message which to draw onto the screen
 */
void Console::drawStringPrimaryAlternate (int x, int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawString(x,y,registeredSkin.getSkinPrimaryTextColorAlternate(),registeredSkin.getSkinPrimaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string at a given position using colours from the provided skin
 * using the main text colour from the secondary window
 * @param[in] x The X position at which to draw the text (cols)
 * @param[in] y The Y position at which to draw the text (rows)
  * @param[in] msg Message which to draw onto the screen
 */
void Console::drawStringSecondary (int x, int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawString(x,y,registeredSkin.getSkinSecondaryTextColor(),registeredSkin.getSkinSecondaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string at a given position using colours from the provided skin
 * using the alternate text colour from the secondary window
 * @param[in] x The X position at which to draw the text (cols)
 * @param[in] y The Y position at which to draw the text (rows)
  * @param[in] msg Message which to draw onto the screen
 */
void Console::drawStringSecondaryAlternate (int x, int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	drawString(x,y,registeredSkin.getSkinSecondaryTextColorAlternate(),registeredSkin.getSkinSecondaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string with the center of the screen at a given Y position 
 * using the main text colour from the primary window
 * @param[in] y The Y position at which to draw the text (rows)
  * @param[in] msg Message which to draw onto the screen
 */
void Console::centerStringPrimary (int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	centerString(y,registeredSkin.getSkinPrimaryTextColor(),registeredSkin.getSkinPrimaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string with the center of the screen at a given Y position 
 * using the alternate text colour from the primary window
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] msg Message which to draw onto the screen
 */
void Console::centerStringPrimaryAlternate (int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	centerString(y,registeredSkin.getSkinPrimaryTextColorAlternate(),registeredSkin.getSkinPrimaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string with the center of the screen at a given Y position 
 * using the main text colour from the primary window
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] msg Message which to draw onto the screen
 */
void Console::centerStringSecondary (int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	centerString(y,registeredSkin.getSkinSecondaryTextColor(),registeredSkin.getSkinSecondaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Draws a text string with the center of the screen at a given Y position 
 * using the alternate text colour from the primary window
 * @param[in] y The Y position at which to draw the text (rows)
 * @param[in] msg Message which to draw onto the screen
 */
void Console::centerStringSecondaryAlternate (int y, const char *msg)
{
	if(!isSkinRegistered())handleSkinRegistrationError();
	centerString(y,registeredSkin.getSkinSecondaryTextColorAlternate(),registeredSkin.getSkinSecondaryBgColor(),msg);
}

//_______________________________________________________________________________
/**
 * Registers a skin to use with the console/skin functions
 * @param[in] skin the skin to register
 */
 void Console::registerSkin(Skin skin)
 {
	registeredSkin = skin;
	skinRegistered = true;
 }
 
 void Console::handleSkinRegistrationError()
 {
	drawString(2,2,Console::RED,Console::BLACK,"SKIN NOT REGISTERED WITH CONSOLE CLASS");
 }
 
 //End
 
